<?php
/**
 * WordPress Feed API
 *
 * Many of the functions used in here belong in The Loop, or The Loop for the
 * Feeds.
 *
 * @package WordPress
 * @subpackage Feed
 */

/**
 * RSS container for the bloginfo function.
 *
 * You can retrieve anything that you can using the get_bloginfo() function.
 * Everything will be stripped of tags and characters converted, when the values
 * are retrieved for use in the feeds.
 *
 * @package WordPress
 * @subpackage Feed
 * @since 1.5.1
 * @uses apply_filters() Calls 'get_bloginfo_rss' hook with two parameters.
 * @see get_bloginfo() For the list of possible values to display.
 *
 * @param string $show See get_bloginfo() for possible values.
 * @return string
 */
function get_bloginfo_rss($show = '') {
	$info = strip_tags(get_bloginfo($show));
	return apply_filters('get_bloginfo_rss', convert_chars($info), $show);
}

/**
 * Display RSS container for the bloginfo function.
 *
 * You can retrieve anything that you can using the get_bloginfo() function.
 * Everything will be stripped of tags and characters converted, when the values
 * are retrieved for use in the feeds.
 *
 * @package WordPress
 * @subpackage Feed
 * @since 0.71
 * @uses apply_filters() Calls 'bloginfo_rss' hook with two parameters.
 * @see get_bloginfo() For the list of possible values to display.
 *
 * @param string $show See get_bloginfo() for possible values.
 */
function bloginfo_rss($show = '') {
	echo apply_filters('bloginfo_rss', get_bloginfo_rss($show), $show);
}

/**
 * Retrieve the default feed.
 *
 * The default feed is 'rss2', unless a plugin changes it through the
 * 'default_feed' filter.
 *
 * @package WordPress
 * @subpackage Feed
 * @since 2.5
 * @uses apply_filters() Calls 'default_feed' hook on the default feed string.
 *
 * @return string Default feed, or for example 'rss2', 'atom', etc.
 */
function get_default_feed() {
	return apply_filters('default_feed', 'rss2');
}

/**
 * Retrieve the blog title for the feed title.
 *
 * @package WordPress
 * @subpackage Feed
 * @since 2.2.0
 * @uses apply_filters() Calls 'get_wp_title_rss' hook on title.
 * @uses wp_title() See function for $sep parameter usage.
 *
 * @param string $sep Optional.How to separate the title. See wp_title() for more info.
 * @return string Error message on failure or blog title on success.
 */
function get_wp_title_rss($sep = '&#187;') {
	$title = wp_title($sep, false);
	if ( is_wp_error( $title ) )
		return $title->get_error_message();
	$title = apply_filters('get_wp_title_rss', $title);
	return $title;
}

/**
 * Display the blog title for display of the feed title.
 *
 * @package WordPress
 * @subpackage Feed
 * @since 2.2.0
 * @uses apply_filters() Calls 'wp_title_rss' on the blog title.
 * @see wp_title() $sep parameter usage.
 *
 * @param string $sep Optional.
 */
function wp_title_rss($sep = '&#187;') {
	echo apply_filters('wp_title_rss', get_wp_title_rss($sep));
}

/**
 * Retrieve the current post title for the feed.
 *
 * @package WordPress
 * @subpackage Feed
 * @since 2.0.0
 * @uses apply_filters() Calls 'the_title_rss' on the post title.
 *
 * @return string Current post title.
 */
function get_the_title_rss() {
	$title = get_the_title();
	$title = apply_filters('the_title_rss', $title);
	return $title;
}

/**
 * Display the post title in the feed.
 *
 * @package WordPress
 * @subpackage Feed
 * @since 0.71
 * @uses get_the_title_rss() Used to retrieve current post title.
 */
function the_title_rss() {
	echo get_the_title_rss();
}

/**
 * Retrieve the post content for feeds.
 *
 * @package WordPress
 * @subpackage Feed
 * @since 2.9.0
 * @uses apply_filters() Calls 'the_content_feed' on the content before processing.
 * @see get_the_content()
 *
 * @param string $feed_type The type of feed. rss2 | atom | rss | rdf
 */
function get_the_content_feed($feed_type = null) {
	if ( !$feed_type )
		$feed_type = get_default_feed();

	$content = apply_filters('the_content', get_the_content());
	$content = str_replace(']]>', ']]&gt;', $content);
	return apply_filters('the_content_feed', $content, $feed_type);
}

/**
 * Display the post content for feeds.
 *
 * @package WordPress
 * @subpackage Feed
 * @since 2.9.0
 * @uses apply_filters() Calls 'the_content_feed' on the content before processing.
 * @see get_the_content()
 *
 * @param string $feed_type The type of feed. rss2 | atom | rss | rdf
 */
function the_content_feed($feed_type = null) {
	echo get_the_content_feed();
}

/**
 * Display the post excerpt for the feed.
 *
 * @package WordPress
 * @subpackage Feed
 * @since 0.71
 * @uses apply_filters() Calls 'the_excerpt_rss' hook on the excerpt.
 */
function the_excerpt_rss() {
	$output = get_the_excerpt();
	echo apply_filters('the_excerpt_rss', $output);
}

/**
 * Display the permalink to the post for use in feeds.
 *
 * @package WordPress
 * @subpackage Feed
 * @since 2.3.0
 * @uses apply_filters() Call 'the_permalink_rss' on the post permalink
 */
function the_permalink_rss() {
	echo apply_filters('the_permalink_rss', get_permalink());
}

/**
 * Display the feed GUID for the current comment.
 *
 * @package WordPress
 * @subpackage Feed
 * @since unknown
 *
 * @param int|object $comment_id Optional comment object or id. Defaults to global comment object.
 */
function comment_guid($comment_id = null) {
	echo get_comment_guid($comment_id);
}

/**
 * Retrieve the feed GUID for the current comment.
 *
 * @package WordPress
 * @subpackage Feed
 * @since unknown
 *
 * @param int|object $comment_id Optional comment object or id. Defaults to global comment object.
 * @return bool|string false on failure or guid for comment on success.
 */
function get_comment_guid($comment_id = null) {
	$comment = get_comment($comment_id);

	if ( !is_object($comment) )
		return false;

	return get_the_guid($comment->comment_p